from sqlalchemy import Boolean, Column, Integer, String, DateTime, Text, JSON, ForeignKey, Float, BigInteger
from sqlalchemy.orm import relationship
from sqlalchemy.sql import func
from app.database import Base


class Task(Base):
    __tablename__ = "tasks"
    
    id = Column(String(36), primary_key=True, index=True)
    user_id = Column(Integer, ForeignKey("users.id", ondelete="CASCADE"))
    title = Column(String(200), nullable=False)
    description = Column(Text)
    goal = Column(Text, nullable=False)
    task_type = Column(String(50), default="general")  # desktop, web, analysis, etc.
    
    # وضعیت
    status = Column(String(20), default="pending")  # pending, queued, processing, completed, failed, cancelled
    progress = Column(Integer, default=0)  # 0-100
    risk_level = Column(String(20), default="low")  # low, medium, high, extreme
    priority = Column(Integer, default=5)  # 1-10
    
    # تنظیمات اجرا
    requires_confirmation = Column(Boolean, default=True)
    max_execution_time = Column(Integer)  # حداکثر زمان اجرا بر حسب ثانیه
    allowed_actions = Column(JSON)  # لیست اقدامات مجاز
    
    # نتیجه
    result_data = Column(JSON)
    result_files = Column(JSON)  # لیست فایل‌های نتیجه
    error_message = Column(Text)
    execution_time = Column(Float)  # در ثانیه
    
    # زمان‌ها
    created_at = Column(DateTime(timezone=True), server_default=func.now())
    queued_at = Column(DateTime(timezone=True))
    started_at = Column(DateTime(timezone=True))
    completed_at = Column(DateTime(timezone=True))
    
    # روابط
    user = relationship("User", back_populates="tasks")
    actions = relationship("Action", back_populates="task", cascade="all, delete-orphan")
    files = relationship("TaskFile", back_populates="task", cascade="all, delete-orphan")
    logs = relationship("TaskLog", back_populates="task", cascade="all, delete-orphan")


class Action(Base):
    __tablename__ = "actions"
    
    id = Column(Integer, primary_key=True, index=True)
    task_id = Column(String(36), ForeignKey("tasks.id", ondelete="CASCADE"))
    action_type = Column(String(50), nullable=False)  # typing, mouse, click, screenshot, etc.
    description = Column(Text, nullable=False)
    
    # پارامترها
    parameters = Column(JSON)
    risk_assessment = Column(JSON)
    
    # وضعیت
    status = Column(String(20), default="pending")  # pending, executing, completed, failed
    result = Column(JSON)
    requires_confirmation = Column(Boolean, default=False)
    confirmed = Column(Boolean, default=False)
    confirmation_code = Column(String(20))
    
    # زمان‌ها
    created_at = Column(DateTime(timezone=True), server_default=func.now())
    confirmed_at = Column(DateTime(timezone=True))
    executed_at = Column(DateTime(timezone=True))
    
    # روابط
    task = relationship("Task", back_populates="actions")


class TaskFile(Base):
    __tablename__ = "task_files"
    
    id = Column(String(36), primary_key=True, index=True)
    task_id = Column(String(36), ForeignKey("tasks.id", ondelete="CASCADE"))
    user_id = Column(Integer, ForeignKey("users.id", ondelete="CASCADE"))
    
    # اطلاعات فایل
    filename = Column(String(255), nullable=False)
    original_filename = Column(String(255), nullable=False)
    file_path = Column(String(500), nullable=False)
    file_size = Column(BigInteger)  # اندازه به بایت
    mime_type = Column(String(100))
    
    # نوع فایل
    file_type = Column(String(50))  # upload, result, screenshot, log, etc.
    is_temp = Column(Boolean, default=True)  # آیا فایل موقت است؟
    expires_at = Column(DateTime(timezone=True))  # تاریخ انقضا
    
    # زمان‌ها
    created_at = Column(DateTime(timezone=True), server_default=func.now())
    
    # روابط
    task = relationship("Task", back_populates="files")
    user = relationship("User", back_populates="files")


class TaskLog(Base):
    __tablename__ = "task_logs"
    
    id = Column(Integer, primary_key=True, index=True)
    task_id = Column(String(36), ForeignKey("tasks.id", ondelete="CASCADE"))
    log_level = Column(String(20), default="info")  # info, warning, error, debug
    message = Column(Text, nullable=False)
    details = Column(JSON)
    
    created_at = Column(DateTime(timezone=True), server_default=func.now())
    
    # روابط
    task = relationship("Task", back_populates="logs")


class TaskQueue(Base):
    __tablename__ = "task_queue"
    
    id = Column(Integer, primary_key=True, index=True)
    task_id = Column(String(36), ForeignKey("tasks.id", ondelete="CASCADE"), unique=True)
    user_id = Column(Integer, ForeignKey("users.id", ondelete="CASCADE"))
    
    # وضعیت در صف
    queue_status = Column(String(20), default="pending")  # pending, processing, completed, failed
    position = Column(Integer)  # موقعیت در صف
    retry_count = Column(Integer, default=0)
    
    # زمان‌ها
    added_to_queue_at = Column(DateTime(timezone=True), server_default=func.now())
    started_processing_at = Column(DateTime(timezone=True))
    completed_at = Column(DateTime(timezone=True))
    
    # اطلاعات اجرا
    worker_id = Column(String(100))  # شناسه کارگر Celery
    node_name = Column(String(100))  # نام نود اجرا
    
    # روابط
    task = relationship("Task")
    user = relationship("User")
