from celery import shared_task
from sqlalchemy.orm import Session
from app.database import SessionLocal
from app.crud import file_crud
import logging
import os
import shutil
from datetime import datetime

logger = logging.getLogger(__name__)


@shared_task
def cleanup_expired_files():
    """
    پاکسازی فایل‌های موقت قدیمی
    این تسک هر 2 ساعت اجرا می‌شود
    """
    db = SessionLocal()
    try:
        # پاکسازی فایل‌های منقضی شده از دیتابیس
        deleted_count = file_crud.cleanup_expired_files(db)
        
        # پاکسازی پوشه‌های موقت
        temp_dirs = ["/app/temp", "/app/uploads/temp"]
        
        for temp_dir in temp_dirs:
            if os.path.exists(temp_dir):
                # حذف فایل‌های قدیمی تر از 2 ساعت
                for filename in os.listdir(temp_dir):
                    file_path = os.path.join(temp_dir, filename)
                    try:
                        # بررسی زمان آخرین تغییر
                        if os.path.isfile(file_path):
                            file_age = datetime.now().timestamp() - os.path.getmtime(file_path)
                            if file_age > 2 * 3600:  # بیشتر از 2 ساعت
                                os.remove(file_path)
                                logger.info(f"فایل قدیمی حذف شد: {file_path}")
                    except Exception as e:
                        logger.error(f"خطا در حذف فایل {file_path}: {e}")
        
        logger.info(f"پاکسازی فایل‌ها کامل شد. {deleted_count} فایل حذف شد.")
        
        return {"deleted_count": deleted_count}
    
    except Exception as e:
        logger.error(f"خطا در پاکسازی فایل‌ها: {e}")
        raise
    
    finally:
        db.close()


@shared_task
def cleanup_old_task_data():
    """
    پاکسازی داده‌های قدیمی تسک‌ها
    تسک‌های تکمیل شده قدیمی‌تر از 30 روز
    """
    db = SessionLocal()
    try:
        from app.models import Task
        from datetime import datetime, timedelta
        
        cutoff_date = datetime.utcnow() - timedelta(days=30)
        
        # یافتن تسک‌های قدیمی
        old_tasks = db.query(Task).filter(
            Task.status.in_(["completed", "failed", "cancelled"]),
            Task.completed_at < cutoff_date
        ).all()
        
        deleted_count = 0
        for task in old_tasks:
            try:
                # حذف فایل‌های مرتبط
                for file in task.files:
                    if os.path.exists(file.file_path):
                        os.remove(file.file_path)
                
                # حذف تسک از دیتابیس
                db.delete(task)
                deleted_count += 1
                
            except Exception as e:
                logger.error(f"خطا در حذف تسک {task.id}: {e}")
        
        db.commit()
        
        if deleted_count > 0:
            logger.info(f"تسک‌های قدیمی پاکسازی شدند: {deleted_count} تسک حذف شد")
        
        return {"deleted_tasks": deleted_count}
        
    except Exception as e:
        logger.error(f"خطا در پاکسازی تسک‌های قدیمی: {e}")
        db.rollback()
        raise
    
    finally:
        db.close()
